require('dotenv').config();
const express = require('express');
const bodyParser = require('body-parser');
const cookieParser = require('cookie-parser');
const session = require('express-session');
const path = require('path');
const multer = require('multer');
const fs = require('fs');
const methodOverride = require('method-override');
const { v4: uuidv4 } = require('uuid');
const moment = require('moment');
const bcrypt = require('bcryptjs');
const { db, dbHelpers } = require('./database');

const app = express();
const PORT = process.env.PORT || 3000;

// Fungsi untuk auto-create admin dari environment variables (deprecated)
// Sekarang menggunakan setup manual melalui /setup-admin
async function autoCreateAdmin() {
    console.log('ℹ️  Auto-create admin sudah tidak digunakan. Gunakan /setup-admin untuk setup pertama kali.');
}

// Middleware
app.use(bodyParser.urlencoded({ extended: true }));
app.use(bodyParser.json());
app.use(methodOverride('_method'));
app.use(methodOverride(function (req, res) {
    if (req.query && req.query._method) {
        return req.query._method;
    }
}));
app.use(cookieParser());
app.use(session({
    secret: process.env.SESSION_SECRET || 'change-this-in-production',
    resave: false,
    saveUninitialized: true,
    cookie: { secure: false }
}));

// Set view engine
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// Static files
app.use(express.static(path.join(__dirname, 'public')));

// Buat folder uploads jika belum ada
const uploadsDir = path.join(__dirname, 'public', 'uploads');
if (!fs.existsSync(uploadsDir)) {
    fs.mkdirSync(uploadsDir, { recursive: true });
}

// Konfigurasi multer untuk upload gambar
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, uploadsDir);
    },
    filename: function (req, file, cb) {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, 'product-' + uniqueSuffix + path.extname(file.originalname));
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB
    },
    fileFilter: function (req, file, cb) {
        if (file.mimetype.startsWith('image/')) {
            cb(null, true);
        } else {
            cb(new Error('Hanya file gambar yang diperbolehkan!'), false);
        }
    }
});

// Middleware untuk handle upload error
const handleUploadError = (err, req, res, next) => {
    if (err instanceof multer.MulterError) {
        if (err.code === 'LIMIT_FILE_SIZE') {
            return res.redirect('/products?error=File terlalu besar. Maksimal 5MB.');
        }
    } else if (err) {
        return res.redirect('/products?error=' + err.message);
    }
    next();
};

// Middleware untuk cek apakah admin sudah ada
async function checkAdminExists(req, res, next) {
    try {
        const hasAdmin = await dbHelpers.hasAdminUser();
        if (!hasAdmin && req.path !== '/setup-admin' && req.path !== '/') {
            return res.redirect('/setup-admin');
        }
        next();
    } catch (error) {
        console.error('Error checking admin:', error);
        next();
    }
}

// Middleware untuk cek login
function requireLogin(req, res, next) {
    if (req.session.user) {
        next();
    } else {
        res.redirect('/login');
    }
}

// Database sudah diinisialisasi di database.js

// Routes
app.get('/', async (req, res) => {
    try {
        // Cek apakah admin sudah ada
        const hasAdmin = await dbHelpers.hasAdminUser();
        if (!hasAdmin) {
            return res.redirect('/setup-admin');
        }
        
        const products = await dbHelpers.getAllProducts();
        res.render('index', {
            products: products,
            user: req.session.user
        });
    } catch (error) {
        console.error('Error fetching products:', error);
        res.render('index', {
            products: [],
            user: req.session.user
        });
    }
});
// Route setup admin (first time setup)
app.get('/setup-admin', async (req, res) => {
    try {
        const hasAdmin = await dbHelpers.hasAdminUser();
        if (hasAdmin) {
            return res.redirect('/login');
        }
        res.render('setup-admin', { error: null, success: null });
    } catch (error) {
        console.error('Error checking admin:', error);
        res.render('setup-admin', { error: 'Terjadi kesalahan sistem', success: null });
    }
});

// Proses setup admin
app.post('/setup-admin', async (req, res) => {
    try {
        const { fullName, username, email, password, confirmPassword } = req.body;
        
        // Validasi input
        if (!fullName || !username || !email || !password || !confirmPassword) {
            return res.render('setup-admin', { 
                error: 'Semua field harus diisi', 
                success: null 
            });
        }
        
        if (password !== confirmPassword) {
            return res.render('setup-admin', { 
                error: 'Password dan konfirmasi password tidak cocok', 
                success: null 
            });
        }
        
        if (password.length < 6) {
            return res.render('setup-admin', { 
                error: 'Password minimal 6 karakter', 
                success: null 
            });
        }
        
        // Cek apakah admin sudah ada
        const hasAdmin = await dbHelpers.hasAdminUser();
        if (hasAdmin) {
            return res.redirect('/login');
        }
        
        // Hash password
        const hashedPassword = await bcrypt.hash(password, 10);
        
        // Buat user admin
        const userId = await dbHelpers.createUser({
            username: username,
            email: email,
            password: hashedPassword,
            fullName: fullName,
            role: 'admin'
        });
        
        console.log(`✅ Admin berhasil dibuat: ${username} (${email})`);
        
        // Set session dan redirect ke dashboard
        req.session.user = {
            id: userId,
            username: username,
            email: email,
            fullName: fullName,
            role: 'admin'
        };
        
        res.redirect('/admin');
        
    } catch (error) {
        console.error('Error creating admin:', error);
        res.render('setup-admin', { 
            error: 'Gagal membuat admin. Silakan coba lagi.', 
            success: null 
        });
    }
});

// Route login
app.get('/login', async (req, res) => {
    try {
        const hasAdmin = await dbHelpers.hasAdminUser();
        if (!hasAdmin) {
            return res.redirect('/setup-admin');
        }
        res.render('login', { error: null });
    } catch (error) {
        console.error('Error checking admin:', error);
        res.render('login', { error: null });
    }
 });

// Register page
app.get('/register', (req, res) => {
    res.render('register', { error: null, success: null });
});

// Register process
app.post('/register', async (req, res) => {
    const { username, email, password, confirmPassword, fullName, phone, website, socialMedia, bankAccount, agreeTerms } = req.body;
    
    // Debug logging
    console.log('Registration attempt:', {
        username,
        email,
        fullName,
        phone,
        agreeTerms,
        agreeTermsType: typeof agreeTerms
    });
    
    try {
        // Validation
        if (!username || !email || !password || !confirmPassword || !fullName || !phone || !bankAccount) {
            return res.render('register', {
                error: 'Semua field wajib harus diisi!',
                success: null
            });
        }
        
        if (password !== confirmPassword) {
            return res.render('register', {
                error: 'Password dan konfirmasi password tidak cocok!',
                success: null
            });
        }
        
        if (password.length < 6) {
            return res.render('register', {
                error: 'Password minimal 6 karakter!',
                success: null
            });
        }
        
        // Check if username or email already exists
        const existingUsername = await dbHelpers.getUserByUsername(username);
        const existingEmail = await dbHelpers.getUserByEmail(email);
        
        if (existingUsername || existingEmail) {
            return res.render('register', {
                error: 'Username atau email sudah terdaftar!',
                success: null
            });
        }
        
        if (!agreeTerms || agreeTerms !== 'on') {
            return res.render('register', {
                error: 'Anda harus menyetujui syarat dan ketentuan!',
                success: null
            });
        }
        
        // Create new user
        const userData = {
            username,
            email,
            password,
            fullName,
            phone,
            website: website || '',
            socialMedia: socialMedia || '',
            bankAccount,
            role: 'affiliate',
            status: 'pending'
        };
        
        await dbHelpers.createUser(userData);
        
        res.render('register', {
            error: null,
            success: 'Pendaftaran berhasil! Silakan tunggu persetujuan admin untuk dapat login.'
        });
    } catch (error) {
        console.error('Error during registration:', error);
        res.render('register', {
            error: 'Terjadi kesalahan saat mendaftar. Silakan coba lagi.',
            success: null
        });
    }
});

app.post('/login', async (req, res) => {
    const { username, password } = req.body;
    
    try {
        const user = await dbHelpers.getUserByUsername(username);
        
        if (user && bcrypt.compareSync(password, user.password)) {
            if (user.status !== 'active') {
                return res.render('login', { error: 'Akun Anda belum diaktifkan atau sedang pending approval!' });
            }
            req.session.user = user;
            res.redirect('/dashboard');
        } else {
            res.render('login', { error: 'Username atau password salah!' });
        }
    } catch (error) {
        console.error('Error during login:', error);
        res.render('login', { error: 'Terjadi kesalahan saat login. Silakan coba lagi.' });
    }
});

app.get('/logout', (req, res) => {
    req.session.destroy();
    res.redirect('/');
});

app.get('/dashboard', checkAdminExists, async (req, res) => {
    if (!req.session.user) {
        return res.redirect('/login');
    }
    
    try {
        const products = await dbHelpers.getAllProducts();
        const analytics = await dbHelpers.getAllAnalytics();
        
        // Calculate statistics
        const totalProducts = products.length;
        const totalClicks = analytics.filter(a => a.action === 'click').length;
        const totalConversions = analytics.filter(a => a.action === 'conversion').length;
        const totalEarnings = products.reduce((sum, product) => sum + product.earnings, 0);
        
        res.render('dashboard', {
            user: req.session.user,
            products: products,
            analytics: analytics,
            stats: {
                totalProducts,
                totalClicks,
                totalConversions,
                totalEarnings
            }
        });
    } catch (error) {
        console.error('Error fetching dashboard data:', error);
        res.render('dashboard', {
            user: req.session.user,
            products: [],
            analytics: [],
            stats: {
                totalProducts: 0,
                totalClicks: 0,
                totalConversions: 0,
                totalEarnings: 0
            }
        });
    }
});

app.get('/products', checkAdminExists, async (req, res) => {
    if (!req.session.user) {
        return res.redirect('/login');
    }
    
    try {
        const products = await dbHelpers.getAllProducts();
        res.render('products', {
            user: req.session.user,
            products: products
        });
    } catch (error) {
        console.error('Error fetching products:', error);
        res.render('products', {
            user: req.session.user,
            products: []
        });
    }
});

app.post('/products', upload.single('productImage'), handleUploadError, async (req, res) => {
    if (!req.session.user || req.session.user.role !== 'admin') {
        return res.redirect('/login');
    }
    
    console.log('Request body:', req.body);
    console.log('Request file:', req.file);
    
    const { name, category, description, price, commission, affiliateLink } = req.body;
    
    // Validate required fields
    if (!name || !category || !description || !price || !commission || !affiliateLink) {
        console.error('Missing required fields:', { name, category, description, price, commission, affiliateLink });
        return res.redirect('/products?error=missing_fields');
    }
    
    try {
        const productData = {
            name: name.trim(),
            category: category.trim(),
            description: description.trim(),
            price: parseFloat(price),
            commission: parseFloat(commission),
            affiliateLink: affiliateLink.trim(),
            base_url: req.body.base_url ? req.body.base_url.trim() : null,
            image: req.file ? '/uploads/' + req.file.filename : null
        };
        
        console.log('Product data to save:', productData);
        
        const newProduct = await dbHelpers.createProduct(productData);
        res.redirect('/products?success=product_created');
    } catch (error) {
        console.error('Error creating product:', error);
        res.redirect('/products?error=creation_failed');
    }
});

app.delete('/products/:id', async (req, res) => {
    if (!req.session.user || req.session.user.role !== 'admin') {
        return res.status(403).json({ error: 'Unauthorized' });
    }
    
    try {
        const result = await dbHelpers.deleteProduct(req.params.id);
        if (result.changes > 0) {
            res.json({ success: true });
        } else {
            res.status(404).json({ error: 'Product not found' });
        }
    } catch (error) {
        console.error('Error deleting product:', error);
        res.status(500).json({ error: 'Terjadi kesalahan saat menghapus produk' });
    }
});

// Get product data for editing
app.get('/products/:id/edit', async (req, res) => {
    if (!req.session.user || req.session.user.role !== 'admin') {
        return res.status(403).json({ success: false, message: 'Access denied' });
    }
    
    try {
        const productId = req.params.id;
        const product = await dbHelpers.getProductById(productId);
        
        if (!product) {
            return res.status(404).json({ success: false, message: 'Product not found' });
        }
        
        res.json({ success: true, product: product });
    } catch (error) {
        console.error('Error fetching product for edit:', error);
        res.status(500).json({ success: false, message: 'Internal server error' });
    }
});

// Update product
app.put('/products/:id', upload.single('productImage'), handleUploadError, async (req, res) => {
    if (!req.session.user || req.session.user.role !== 'admin') {
        return res.status(403).json({ success: false, message: 'Access denied' });
    }
    
    try {
        const productId = req.params.id;
        const { name, category, description, price, commission, affiliateLink, base_url } = req.body;
        
        // Get current product data to preserve existing image if no new image uploaded
        const currentProduct = await dbHelpers.getProductById(productId);
        
        const productData = {
            name,
            category,
            description,
            price: parseFloat(price),
            commission: parseFloat(commission),
            affiliateLink,
            base_url: base_url ? base_url.trim() : null
        };
        
        // Add image if uploaded, otherwise keep existing image
        if (req.file) {
            productData.image = '/uploads/' + req.file.filename;
        } else if (currentProduct && currentProduct.image) {
            productData.image = currentProduct.image;
        }
        
        const result = await dbHelpers.updateProduct(productId, productData);
        
        if (result.changes > 0) {
            res.redirect('/products');
        } else {
            res.status(404).json({ success: false, message: 'Product not found' });
        }
    } catch (error) {
        console.error('Error updating product:', error);
        res.status(500).json({ success: false, message: 'Internal server error' });
    }
});

app.get('/track/:id', async (req, res) => {
    const productId = req.params.id;
    
    try {
        const product = await dbHelpers.getProductById(productId);
        
        if (!product) {
            return res.status(404).send('Product not found');
        }
        
        // Track click
        await dbHelpers.createAnalytics({
            productId: productId,
            action: 'click',
            userAgent: req.get('User-Agent'),
            ip: req.ip
        });
        
        // Update product clicks
        await dbHelpers.updateProductClicks(productId, product.clicks + 1);
        
        // Redirect to affiliate link
        res.redirect(product.affiliateLink);
    } catch (error) {
        console.error('Error tracking click:', error);
        res.status(500).send('Terjadi kesalahan');
    }
});

app.post('/convert/:id', (req, res) => {
    const productId = req.params.id;
    const product = products.find(p => p.id === productId);
    
    if (product) {
        product.conversions++;
        product.earnings += (product.price * product.commission / 100);
        
        analytics.push({
            id: uuidv4(),
            productId: productId,
            action: 'conversion',
            timestamp: new Date(),
            ip: req.ip,
            userAgent: req.get('User-Agent')
        });
        
        res.json({ success: true });
    } else {
        res.status(404).json({ error: 'Product not found' });
    }
});

app.get('/analytics', checkAdminExists, async (req, res) => {
    if (!req.session.user) {
        return res.redirect('/login');
    }
    
    try {
        const products = await dbHelpers.getAllProducts();
        const analytics = await dbHelpers.getAllAnalytics();
        
        res.render('analytics', {
            user: req.session.user,
            products: products,
            analytics: analytics
        });
    } catch (error) {
        console.error('Error fetching analytics data:', error);
        res.render('analytics', {
            user: req.session.user,
            products: [],
            analytics: []
        });
    }
});

// Users management (Admin only)
app.get('/users', checkAdminExists, async (req, res) => {
    if (!req.session.user || req.session.user.role !== 'admin') {
        return res.redirect('/login');
    }
    
    try {
        const users = await dbHelpers.getAllUsers();
        res.render('users', {
            user: req.session.user,
            users: users
        });
    } catch (error) {
        console.error('Error fetching users:', error);
        res.render('users', {
            user: req.session.user,
            users: []
        });
    }
});

// Approve user
app.post('/users/:id/approve', async (req, res) => {
    if (!req.session.user || req.session.user.role !== 'admin') {
        return res.status(403).json({ error: 'Unauthorized' });
    }
    
    try {
        const result = await dbHelpers.updateUserStatus(req.params.id, 'active');
        if (result.changes > 0) {
            res.json({ success: true });
        } else {
            res.status(404).json({ error: 'User not found' });
        }
    } catch (error) {
        console.error('Error approving user:', error);
        res.status(500).json({ error: 'Terjadi kesalahan saat menyetujui user' });
    }
});

// Toggle user status
app.post('/users/:id/toggle', async (req, res) => {
    if (!req.session.user || req.session.user.role !== 'admin') {
        return res.status(403).json({ error: 'Unauthorized' });
    }
    
    try {
        const user = await dbHelpers.getUserById(req.params.id);
        if (!user) {
            return res.status(404).json({ error: 'User not found' });
        }
        
        const newStatus = user.status === 'active' ? 'inactive' : 'active';
        const result = await dbHelpers.updateUserStatus(req.params.id, newStatus);
        
        if (result.changes > 0) {
            res.json({ success: true, status: newStatus });
        } else {
            res.status(500).json({ error: 'Gagal mengubah status user' });
        }
    } catch (error) {
        console.error('Error toggling user status:', error);
        res.status(500).json({ error: 'Terjadi kesalahan saat mengubah status user' });
    }
});

// Custom Links Routes
app.post('/custom-links', async (req, res) => {
    if (!req.session.user) {
        return res.status(401).json({ success: false, message: 'Unauthorized' });
    }
    
    try {
        const { product_id, base_url, custom_slug } = req.body;
        
        // Validate input
        if (!product_id || !base_url || !custom_slug) {
            return res.status(400).json({ success: false, message: 'Semua field harus diisi' });
        }
        
        // Check if slug already exists
        const existingLink = await dbHelpers.getCustomLinkBySlug(custom_slug);
        if (existingLink) {
            return res.status(400).json({ success: false, message: 'Slug sudah digunakan, pilih slug lain' });
        }
        
        // Get product to verify it exists and get tracking link
        const product = await dbHelpers.getProductById(product_id);
        if (!product) {
            return res.status(404).json({ success: false, message: 'Produk tidak ditemukan' });
        }
        
        // Create custom link
        const linkData = {
            product_id: parseInt(product_id),
            user_id: req.session.user.id,
            custom_slug: custom_slug.trim(),
            original_url: `${req.protocol}://${req.get('host')}/track/${product_id}`
        };
        
        const newLink = await dbHelpers.createCustomLink(linkData);
        
        // Return success with the generated link
        const customLink = `${base_url}/${custom_slug}`;
        res.json({ 
            success: true, 
            link: customLink,
            id: newLink.id
        });
        
    } catch (error) {
        console.error('Error creating custom link:', error);
        res.status(500).json({ success: false, message: 'Terjadi kesalahan saat membuat link custom' });
    }
});

// Handle custom link redirects
app.get('/c/:slug', async (req, res) => {
    try {
        const slug = req.params.slug;
        
        // Find custom link by slug
        const customLink = await dbHelpers.getCustomLinkBySlug(slug);
        
        if (!customLink) {
            return res.status(404).send('Link tidak ditemukan');
        }
        
        // Update click count for custom link
        await dbHelpers.updateCustomLinkClicks(customLink.id);
        
        // Redirect to original tracking link
        res.redirect(customLink.original_url);
        
    } catch (error) {
        console.error('Error handling custom link redirect:', error);
        res.status(500).send('Terjadi kesalahan');
    }
});

// Handle cross-domain redirect from main website
// This endpoint receives redirects from www.azkazamdigital.com
app.get('/redirect/:slug', async (req, res) => {
    try {
        const slug = req.params.slug;
        
        // Find custom link by slug
        const customLink = await dbHelpers.getCustomLinkBySlug(slug);
        
        if (!customLink) {
            return res.status(404).send('Link tidak ditemukan');
        }
        
        // Update click count for custom link
        await dbHelpers.updateCustomLinkClicks(customLink.id);
        
        // Redirect to original tracking link
        res.redirect(customLink.original_url);
        
    } catch (error) {
        console.error('Error handling cross-domain redirect:', error);
        res.status(500).send('Terjadi kesalahan');
    }
});

// Get user's custom links
app.get('/my-links', checkAdminExists, async (req, res) => {
    if (!req.session.user) {
        return res.redirect('/login');
    }
    
    try {
        const customLinks = await dbHelpers.getCustomLinksByUser(req.session.user.id);
        res.render('my-links', { 
            user: req.session.user, 
            customLinks,
            moment: require('moment')
        });
    } catch (error) {
        console.error('Error getting custom links:', error);
        res.redirect('/dashboard?error=fetch_links_failed');
    }
});

// Delete custom link
app.delete('/custom-links/:id', async (req, res) => {
    if (!req.session.user) {
        return res.status(401).json({ success: false, message: 'Unauthorized' });
    }
    
    try {
        const result = await dbHelpers.deleteCustomLink(req.params.id, req.session.user.id);
        
        if (result.changes > 0) {
            res.json({ success: true });
        } else {
            res.status(404).json({ success: false, message: 'Link tidak ditemukan atau tidak memiliki akses' });
        }
    } catch (error) {
        console.error('Error deleting custom link:', error);
        res.status(500).json({ success: false, message: 'Terjadi kesalahan saat menghapus link' });
    }
});

// Konfigurasi untuk cPanel - tambahkan Application URL prefix jika ada
const APP_URL_PREFIX = process.env.APP_URL_PREFIX || '';

// Helper function untuk menambahkan prefix pada route
function addPrefix(route) {
    return APP_URL_PREFIX ? `${APP_URL_PREFIX}${route}` : route;
}

// Route create-admin-web sudah tidak digunakan
// Gunakan /setup-admin untuk setup admin yang baru

// API endpoint untuk membuat admin user
app.post(addPrefix('/create-admin-api'), async (req, res) => {
    try {
        const { username, password, email, fullName } = req.body;
        
        // Validasi input
        if (!username || !password || !email || !fullName) {
            return res.json({ 
                success: false, 
                message: 'Semua field harus diisi!' 
            });
        }
        
        // Cek apakah user sudah ada
        const existingUser = await dbHelpers.getUserByUsername(username);
        if (existingUser) {
            return res.json({ 
                success: false, 
                message: 'Username sudah ada! Gunakan username lain atau hapus user yang ada.' 
            });
        }
        
        // Buat user admin (password akan di-hash di createUser function)
        const userData = {
            username: username,
            password: password, // Kirim password asli, akan di-hash di createUser
            email: email,
            fullName: fullName,
            phone: '08123456789',
            website: '',
            socialMedia: '',
            bankAccount: '',
            role: 'admin',
            status: 'active',
            joinDate: new Date().toISOString(),
            totalEarnings: 0
        };
        
        await dbHelpers.createUser(userData);
        
        res.json({ 
            success: true, 
            message: 'Admin user berhasil dibuat!',
            username: username,
            password: password, // Kirim password asli untuk ditampilkan
            email: email
        });
        
    } catch (error) {
        console.error('Error creating admin:', error);
        res.json({ 
            success: false, 
            message: 'Terjadi kesalahan: ' + error.message 
        });
    }
});

// Tambahan endpoint tanpa prefix untuk kompatibilitas
app.post('/create-admin-api', async (req, res) => {
    try {
        const { username, password, email, fullName } = req.body;
        
        // Validasi input
        if (!username || !password || !email || !fullName) {
            return res.json({ 
                success: false, 
                message: 'Semua field harus diisi!' 
            });
        }
        
        // Cek apakah user sudah ada
        const existingUser = await dbHelpers.getUserByUsername(username);
        if (existingUser) {
            return res.json({ 
                success: false, 
                message: 'Username sudah ada! Gunakan username lain atau hapus user yang ada.' 
            });
        }
        
        // Buat user admin (password akan di-hash di createUser function)
        const userData = {
            username: username,
            password: password, // Kirim password asli, akan di-hash di createUser
            email: email,
            fullName: fullName,
            phone: '08123456789',
            website: '',
            socialMedia: '',
            bankAccount: '',
            role: 'admin',
            status: 'active',
            joinDate: new Date().toISOString(),
            totalEarnings: 0
        };
        
        await dbHelpers.createUser(userData);
        
        res.json({ 
            success: true, 
            message: 'Admin user berhasil dibuat!',
            username: username,
            password: password, // Kirim password asli untuk ditampilkan
            email: email
        });
        
    } catch (error) {
        console.error('Error creating admin:', error);
        res.json({ 
            success: false, 
            message: 'Terjadi kesalahan: ' + error.message 
        });
    }
});

app.listen(PORT, async () => {
    if (process.env.NODE_ENV !== 'production') {
        console.log(`Server berjalan di http://localhost:${PORT}`);
    }
    
    // Auto-create admin jika diaktifkan via environment variables
    await autoCreateAdmin();
});