# Setup Cross-Domain Redirect untuk Affiliate System

Panduan ini menjelaskan cara mengkonfigurasi redirect dari domain utama (www.azkazamdigital.com) ke aplikasi affiliate (afiliasi.azkazamdigital.com).

## Skenario
- **Domain Utama**: www.azkazamdigital.com
- **Aplikasi Affiliate**: afiliasi.azkazamdigital.com
- **Tujuan**: Link custom dari domain utama redirect ke aplikasi affiliate untuk tracking

## Cara Kerja
1. User mengakses: `https://www.azkazamdigital.com/slugcustom`
2. Server domain utama redirect ke: `https://afiliasi.azkazamdigital.com/redirect/slugcustom`
3. Aplikasi affiliate melakukan tracking dan redirect ke produk

## Setup Redirect

### Opsi 1: Apache (.htaccess)

Buat file `.htaccess` di root directory www.azkazamdigital.com:

```apache
RewriteEngine On

# Redirect custom affiliate links
# Pattern: /[alphanumeric] -> affiliate app
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^([a-zA-Z0-9]+)$ https://afiliasi.azkazamdigital.com/redirect/$1 [R=302,L]

# Exclude common files/folders from redirect
RewriteCond %{REQUEST_URI} !^/(wp-admin|wp-content|wp-includes|admin|css|js|images|assets)
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^([a-zA-Z0-9]+)$ https://afiliasi.azkazamdigital.com/redirect/$1 [R=302,L]
```

### Opsi 2: Nginx

Tambahkan konfigurasi di server block nginx:

```nginx
server {
    listen 80;
    server_name www.azkazamdigital.com;
    
    # Handle affiliate redirects
    location ~ ^/([a-zA-Z0-9]+)$ {
        # Check if it's not a real file or directory
        try_files $uri $uri/ @affiliate_redirect;
    }
    
    location @affiliate_redirect {
        return 302 https://afiliasi.azkazamdigital.com/redirect$uri;
    }
    
    # Normal website handling
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }
}
```

### Opsi 3: Cloudflare Workers

Buat Cloudflare Worker untuk www.azkazamdigital.com:

```javascript
addEventListener('fetch', event => {
  event.respondWith(handleRequest(event.request))
})

async function handleRequest(request) {
  const url = new URL(request.url)
  const path = url.pathname.slice(1) // Remove leading slash
  
  // Check if it's a potential affiliate slug
  if (path && path.match(/^[a-zA-Z0-9]+$/) && path.length >= 3) {
    // Redirect to affiliate app
    return Response.redirect(`https://afiliasi.azkazamdigital.com/redirect/${path}`, 302)
  }
  
  // Handle normal website traffic
  return fetch(request)
}
```

### Opsi 4: PHP Redirect

Buat file `index.php` atau tambahkan di file existing:

```php
<?php
// Get the requested path
$path = trim($_SERVER['REQUEST_URI'], '/');

// Check if it's a potential affiliate slug
if (preg_match('/^[a-zA-Z0-9]+$/', $path) && strlen($path) >= 3) {
    // Check if it's not a real file or directory
    if (!file_exists($path) && !is_dir($path)) {
        // Redirect to affiliate app
        header("Location: https://afiliasi.azkazamdigital.com/redirect/$path", true, 302);
        exit();
    }
}

// Continue with normal website logic
?>
```

## Konfigurasi di Aplikasi Affiliate

### Setting Base URL Produk

Saat menambah/edit produk di admin panel:
1. Isi field **Base URL** dengan: `https://www.azkazamdigital.com`
2. Link yang dihasilkan akan menjadi: `https://www.azkazamdigital.com/slugcustom`
3. Redirect akan otomatis mengarahkan ke aplikasi affiliate untuk tracking

### Contoh Flow

1. **Generate Link Custom**:
   - Base URL: `https://www.azkazamdigital.com`
   - Custom Slug: `roket123`
   - Link Generated: `https://www.azkazamdigital.com/roket123`

2. **User Click Link**:
   - User mengakses: `https://www.azkazamdigital.com/roket123`
   - Server redirect ke: `https://afiliasi.azkazamdigital.com/redirect/roket123`
   - Aplikasi affiliate track click dan redirect ke produk

## Testing

### Test Redirect

1. Buat link custom di aplikasi affiliate
2. Akses link dari domain utama
3. Pastikan redirect berfungsi dan tracking tercatat

### Debug

Jika redirect tidak berfungsi:
1. Cek konfigurasi server (Apache/Nginx)
2. Pastikan slug tidak konflik dengan file/folder existing
3. Cek log server untuk error
4. Test dengan curl:

```bash
curl -I https://www.azkazamdigital.com/testslug
```

## Keamanan

### Validasi Slug
- Hanya alphanumeric characters yang diizinkan
- Minimum 3 karakter untuk menghindari konflik
- Exclude common paths (admin, api, wp-admin, dll)

### Rate Limiting
Pertimbangkan untuk menambahkan rate limiting pada endpoint redirect:

```nginx
limit_req_zone $binary_remote_addr zone=affiliate:10m rate=10r/s;

location @affiliate_redirect {
    limit_req zone=affiliate burst=20 nodelay;
    return 302 https://afiliasi.azkazamdigital.com/redirect$uri;
}
```

## Troubleshooting

### Masalah Umum

1. **404 Error**: Pastikan konfigurasi redirect sudah benar
2. **Infinite Loop**: Cek exclude rules untuk file/folder existing
3. **Slow Response**: Gunakan 302 redirect, bukan 301
4. **SEO Impact**: Gunakan canonical tags di halaman produk

### Monitoring

Pantau metrics berikut:
- Jumlah redirect per hari
- Response time redirect
- Error rate
- Conversion rate dari cross-domain links

## Kesimpulan

Dengan setup ini, Anda dapat:
- Menggunakan domain utama untuk affiliate links
- Maintain tracking di aplikasi affiliate
- Memberikan pengalaman user yang seamless
- Meningkatkan trust dengan menggunakan domain utama

Pilih opsi setup yang sesuai dengan infrastruktur server Anda.