const sqlite3 = require('sqlite3').verbose();
const path = require('path');
const bcrypt = require('bcryptjs');

// Path ke file database
const dbPath = path.join(__dirname, 'affiliate.db');

// Buat koneksi database
const db = new sqlite3.Database(dbPath, (err) => {
    if (err) {
        console.error('Error opening database:', err.message);
    } else {
        if (process.env.NODE_ENV !== 'production') {
            console.log('Connected to SQLite database.');
        }
        initializeDatabase();
    }
});

// Inisialisasi tabel database
function initializeDatabase() {
    // Tabel users
    db.run(`CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE NOT NULL,
        password TEXT NOT NULL,
        email TEXT UNIQUE NOT NULL,
        fullName TEXT NOT NULL,
        phone TEXT,
        website TEXT,
        socialMedia TEXT,
        bankAccount TEXT,
        role TEXT DEFAULT 'affiliate',
        status TEXT DEFAULT 'pending',
        joinDate TEXT NOT NULL,
        totalEarnings REAL DEFAULT 0
    )`);

    // Tabel products
    db.run(`CREATE TABLE IF NOT EXISTS products (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        category TEXT NOT NULL,
        description TEXT,
        price REAL NOT NULL,
        commission REAL NOT NULL,
        affiliateLink TEXT NOT NULL,
        clicks INTEGER DEFAULT 0,
        conversions INTEGER DEFAULT 0,
        earnings REAL DEFAULT 0,
        image TEXT,
        base_url TEXT
    )`);

    // Tambahkan kolom base_url jika belum ada (untuk database yang sudah ada)
    db.run(`ALTER TABLE products ADD COLUMN base_url TEXT`, (err) => {
        if (err && !err.message.includes('duplicate column name')) {
            console.error('Error adding base_url column:', err.message);
        }
    });

    // Tambahkan kolom image jika belum ada (untuk database yang sudah ada)
    db.run(`ALTER TABLE products ADD COLUMN image TEXT`, (err) => {
        if (err && !err.message.includes('duplicate column name')) {
            console.error('Error adding image column:', err.message);
        }
    });

    // Tabel analytics
    db.run(`CREATE TABLE IF NOT EXISTS analytics (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        productId INTEGER,
        action TEXT NOT NULL,
        timestamp TEXT NOT NULL,
        userAgent TEXT,
        ip TEXT,
        FOREIGN KEY (productId) REFERENCES products (id)
    )`);

    // Tabel custom_links untuk menyimpan link custom user
    db.run(`CREATE TABLE IF NOT EXISTS custom_links (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        product_id INTEGER NOT NULL,
        user_id INTEGER NOT NULL,
        custom_slug TEXT NOT NULL UNIQUE,
        original_url TEXT NOT NULL,
        clicks INTEGER DEFAULT 0,
        created_at TEXT NOT NULL,
        FOREIGN KEY (product_id) REFERENCES products (id),
        FOREIGN KEY (user_id) REFERENCES users (id)
    )`);

    // Note: Create admin user manually in production for security

    // Note: Add products manually through admin panel in production

    // Note: Analytics data will be generated from actual user activity
}

// Database helper functions
const dbHelpers = {
    // Users
    getAllUsers: () => {
        return new Promise((resolve, reject) => {
            db.all("SELECT * FROM users ORDER BY joinDate DESC", (err, rows) => {
                if (err) reject(err);
                else resolve(rows);
            });
        });
    },

    getUserByUsername: (username) => {
        return new Promise((resolve, reject) => {
            db.get("SELECT * FROM users WHERE username = ?", [username], (err, row) => {
                if (err) reject(err);
                else resolve(row);
            });
        });
    },

    getUserByEmail: (email) => {
        return new Promise((resolve, reject) => {
            db.get("SELECT * FROM users WHERE email = ?", [email], (err, row) => {
                if (err) reject(err);
                else resolve(row);
            });
        });
    },

    getUserById: (id) => {
        return new Promise((resolve, reject) => {
            db.get("SELECT * FROM users WHERE id = ?", [id], (err, row) => {
                if (err) reject(err);
                else resolve(row);
            });
        });
    },

    updateProductClicks: (id, clicks) => {
        return new Promise((resolve, reject) => {
            db.run('UPDATE products SET clicks = ? WHERE id = ?', [clicks, id], function(err) {
                if (err) reject(err);
                else resolve(this);
            });
        });
    },

    createUser: (userData) => {
        return new Promise((resolve, reject) => {
            const { username, password, email, fullName, phone, website, socialMedia, bankAccount, role = 'affiliate', status = 'pending' } = userData;
            const hashedPassword = bcrypt.hashSync(password, 10);
            const joinDate = new Date().toISOString();
            
            db.run(`INSERT INTO users (username, password, email, fullName, phone, website, socialMedia, bankAccount, role, status, joinDate, totalEarnings)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                [username, hashedPassword, email, fullName, phone, website, socialMedia, bankAccount, role, status, joinDate, 0],
                function(err) {
                    if (err) reject(err);
                    else resolve({ id: this.lastID, ...userData, password: hashedPassword, joinDate, totalEarnings: 0 });
                }
            );
        });
    },

    updateUserStatus: (id, status) => {
        return new Promise((resolve, reject) => {
            db.run("UPDATE users SET status = ? WHERE id = ?", [status, id], function(err) {
                if (err) reject(err);
                else resolve({ changes: this.changes });
            });
        });
    },

    // Products
    getAllProducts: () => {
        return new Promise((resolve, reject) => {
            db.all("SELECT * FROM products ORDER BY id DESC", (err, rows) => {
                if (err) reject(err);
                else resolve(rows);
            });
        });
    },

    getProductById: (id) => {
        return new Promise((resolve, reject) => {
            db.get("SELECT * FROM products WHERE id = ?", [id], (err, row) => {
                if (err) reject(err);
                else resolve(row);
            });
        });
    },

    createProduct: (productData) => {
        return new Promise((resolve, reject) => {
            const { name, category, description, price, commission, affiliateLink, image, base_url } = productData;
            
            db.run(`INSERT INTO products (name, category, description, price, commission, affiliateLink, clicks, conversions, earnings, image, base_url)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                [name, category, description, price, commission, affiliateLink, 0, 0, 0, image || null, base_url || null],
                function(err) {
                    if (err) reject(err);
                    else resolve({ id: this.lastID, ...productData, clicks: 0, conversions: 0, earnings: 0 });
                }
            );
        });
    },

    updateProduct: (id, productData) => {
        return new Promise((resolve, reject) => {
            const { name, category, description, price, commission, affiliateLink, image, base_url } = productData;
            
            // Always update all fields including image and base_url
            const query = `UPDATE products SET name = ?, category = ?, description = ?, price = ?, commission = ?, affiliateLink = ?, image = ?, base_url = ? WHERE id = ?`;
            const params = [name, category, description, price, commission, affiliateLink, image || null, base_url || null, id];
            
            db.run(query, params, function(err) {
                if (err) reject(err);
                else resolve({ changes: this.changes });
            });
        });
    },

    deleteProduct: (id) => {
        return new Promise((resolve, reject) => {
            db.run("DELETE FROM products WHERE id = ?", [id], function(err) {
                if (err) reject(err);
                else resolve({ changes: this.changes });
            });
        });
    },

    updateProductStats: (id, clicks, conversions, earnings) => {
        return new Promise((resolve, reject) => {
            db.run("UPDATE products SET clicks = ?, conversions = ?, earnings = ? WHERE id = ?", 
                [clicks, conversions, earnings, id], 
                function(err) {
                    if (err) reject(err);
                    else resolve({ changes: this.changes });
                }
            );
        });
    },

    // Analytics
    getAllAnalytics: () => {
        return new Promise((resolve, reject) => {
            db.all("SELECT * FROM analytics ORDER BY timestamp DESC", (err, rows) => {
                if (err) reject(err);
                else resolve(rows);
            });
        });
    },

    createAnalytics: (analyticsData) => {
        return new Promise((resolve, reject) => {
            const { productId, action, userAgent, ip } = analyticsData;
            const timestamp = new Date().toISOString();
            
            db.run(`INSERT INTO analytics (productId, action, timestamp, userAgent, ip)
                    VALUES (?, ?, ?, ?, ?)`,
                [productId, action, timestamp, userAgent, ip],
                function(err) {
                    if (err) reject(err);
                    else resolve({ id: this.lastID, ...analyticsData, timestamp });
                }
            );
        });
    },

    getAnalyticsByDateRange: (startDate, endDate) => {
        return new Promise((resolve, reject) => {
            db.all("SELECT * FROM analytics WHERE timestamp BETWEEN ? AND ? ORDER BY timestamp DESC", 
                [startDate, endDate], 
                (err, rows) => {
                    if (err) reject(err);
                    else resolve(rows);
                }
            );
        });
    },

    // Update user profile
    updateUserProfile: (id, userData) => {
        return new Promise((resolve, reject) => {
            const { fullName, phone, website, socialMedia, bankAccount } = userData;
            db.run('UPDATE users SET fullName = ?, phone = ?, website = ?, socialMedia = ?, bankAccount = ? WHERE id = ?',
                [fullName, phone, website, socialMedia, bankAccount, id],
                function(err) {
                    if (err) reject(err);
                    else resolve({ changes: this.changes });
                }
            );
        });
    },

    // Update user earnings
    updateUserEarnings: (id, earnings) => {
        return new Promise((resolve, reject) => {
            db.run('UPDATE users SET totalEarnings = ? WHERE id = ?', [earnings, id], function(err) {
                if (err) reject(err);
                else resolve({ changes: this.changes });
            });
        });
    },

    // Get analytics by product ID
    getAnalyticsByProductId: (productId) => {
        return new Promise((resolve, reject) => {
            db.all('SELECT * FROM analytics WHERE productId = ? ORDER BY timestamp DESC', [productId], (err, rows) => {
                if (err) reject(err);
                else resolve(rows);
            });
        });
    },

    // Get dashboard statistics
    getDashboardStats: () => {
        return new Promise((resolve, reject) => {
            const stats = {};
            
            // Get total users
            db.get('SELECT COUNT(*) as totalUsers FROM users WHERE role = "affiliate"', (err, row) => {
                if (err) {
                    reject(err);
                    return;
                }
                stats.totalUsers = row.totalUsers;
                
                // Get total products
                db.get('SELECT COUNT(*) as totalProducts FROM products', (err, row) => {
                    if (err) {
                        reject(err);
                        return;
                    }
                    stats.totalProducts = row.totalProducts;
                    
                    // Get total clicks
                    db.get('SELECT SUM(clicks) as totalClicks FROM products', (err, row) => {
                        if (err) {
                            reject(err);
                            return;
                        }
                        stats.totalClicks = row.totalClicks || 0;
                        
                        // Get total earnings
                        db.get('SELECT SUM(earnings) as totalEarnings FROM products', (err, row) => {
                            if (err) {
                                reject(err);
                                return;
                            }
                            stats.totalEarnings = row.totalEarnings || 0;
                            resolve(stats);
                        });
                    });
                });
            });
        });
    },

    // Custom Links functions
    createCustomLink: (linkData) => {
        return new Promise((resolve, reject) => {
            const { product_id, user_id, custom_slug, original_url } = linkData;
            const created_at = new Date().toISOString();
            
            db.run(`INSERT INTO custom_links (product_id, user_id, custom_slug, original_url, created_at)
                    VALUES (?, ?, ?, ?, ?)`,
                [product_id, user_id, custom_slug, original_url, created_at],
                function(err) {
                    if (err) {
                        reject(err);
                    } else {
                        resolve({ id: this.lastID, ...linkData, created_at });
                    }
                }
            );
        });
    },

    getCustomLinkBySlug: (slug) => {
        return new Promise((resolve, reject) => {
            db.get('SELECT * FROM custom_links WHERE custom_slug = ?', [slug], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });
    },

    getCustomLinksByUser: (userId) => {
        return new Promise((resolve, reject) => {
            db.all(`SELECT cl.*, p.name as product_name, p.base_url 
                    FROM custom_links cl 
                    JOIN products p ON cl.product_id = p.id 
                    WHERE cl.user_id = ? 
                    ORDER BY cl.created_at DESC`, [userId], (err, rows) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(rows);
                }
            });
        });
    },

    updateCustomLinkClicks: (id) => {
        return new Promise((resolve, reject) => {
            db.run('UPDATE custom_links SET clicks = clicks + 1 WHERE id = ?', [id], function(err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({ changes: this.changes });
                }
            });
        });
    },

    deleteCustomLink: (id, userId) => {
        return new Promise((resolve, reject) => {
            db.run(
                'DELETE FROM custom_links WHERE id = ? AND user_id = ?',
                [id, userId],
                function(err) {
                    if (err) {
                        reject(err);
                    } else {
                        resolve({ changes: this.changes });
                    }
                }
            );
        });
    },

    // Fungsi untuk cek apakah admin sudah ada
    hasAdminUser: () => {
        return new Promise((resolve, reject) => {
            db.get(
                'SELECT COUNT(*) as count FROM users WHERE role = "admin"',
                [],
                (err, row) => {
                    if (err) {
                        reject(err);
                    } else {
                        resolve(row.count > 0);
                    }
                }
            );
        });
    }
};

module.exports = { db, dbHelpers };