// Main JavaScript for Afiliasi Azkazamdigital

// Initialize application when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

function initializeApp() {
    // Add fade-in animation to cards
    animateCards();
    
    // Initialize tooltips
    initializeTooltips();
    
    // Initialize form validation
    initializeFormValidation();
    
    // Initialize click tracking
    initializeClickTracking();
    
    // Initialize auto-refresh for dashboard
    initializeAutoRefresh();
}

// Animate cards on page load
function animateCards() {
    const cards = document.querySelectorAll('.card');
    cards.forEach((card, index) => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            card.style.transition = 'all 0.6s ease';
            card.style.opacity = '1';
            card.style.transform = 'translateY(0)';
        }, index * 100);
    });
}

// Initialize Bootstrap tooltips
function initializeTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
}

// Form validation
function initializeFormValidation() {
    const forms = document.querySelectorAll('.needs-validation');
    
    Array.prototype.slice.call(forms).forEach(function (form) {
        form.addEventListener('submit', function (event) {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            form.classList.add('was-validated');
        }, false);
    });
    
    // Real-time validation for product form
    const productForm = document.querySelector('#addProductModal form');
    if (productForm) {
        const inputs = productForm.querySelectorAll('input, textarea, select');
        inputs.forEach(input => {
            input.addEventListener('blur', validateField);
            input.addEventListener('input', clearValidation);
        });
    }
}

function validateField(event) {
    const field = event.target;
    const value = field.value.trim();
    
    // Remove existing validation classes
    field.classList.remove('is-valid', 'is-invalid');
    
    // Validate based on field type
    let isValid = true;
    let errorMessage = '';
    
    switch (field.name) {
        case 'name':
            isValid = value.length >= 3;
            errorMessage = 'Nama produk minimal 3 karakter';
            break;
        case 'description':
            isValid = value.length >= 10;
            errorMessage = 'Deskripsi minimal 10 karakter';
            break;
        case 'price':
            isValid = parseFloat(value) > 0;
            errorMessage = 'Harga harus lebih dari 0';
            break;
        case 'commission':
            const commission = parseFloat(value);
            isValid = commission >= 0 && commission <= 100;
            errorMessage = 'Komisi harus antara 0-100%';
            break;
        case 'affiliateLink':
            isValid = isValidURL(value);
            errorMessage = 'URL tidak valid';
            break;
    }
    
    // Apply validation classes
    if (isValid) {
        field.classList.add('is-valid');
        removeErrorMessage(field);
    } else {
        field.classList.add('is-invalid');
        showErrorMessage(field, errorMessage);
    }
}

function clearValidation(event) {
    const field = event.target;
    field.classList.remove('is-valid', 'is-invalid');
    removeErrorMessage(field);
}

function showErrorMessage(field, message) {
    removeErrorMessage(field);
    
    const errorDiv = document.createElement('div');
    errorDiv.className = 'invalid-feedback';
    errorDiv.textContent = message;
    
    field.parentNode.appendChild(errorDiv);
}

function removeErrorMessage(field) {
    const existingError = field.parentNode.querySelector('.invalid-feedback');
    if (existingError) {
        existingError.remove();
    }
}

function isValidURL(string) {
    try {
        new URL(string);
        return true;
    } catch (_) {
        return false;
    }
}

// Click tracking for analytics
function initializeClickTracking() {
    const trackingLinks = document.querySelectorAll('a[href*="/track/"]');
    
    trackingLinks.forEach(link => {
        link.addEventListener('click', function(event) {
            // Add visual feedback
            const originalText = this.innerHTML;
            this.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Redirecting...';
            this.classList.add('disabled');
            
            // Track the click (optional: send to analytics)
            const productId = this.href.split('/track/')[1];
            trackClick(productId);
            
            // Restore original state after delay
            setTimeout(() => {
                this.innerHTML = originalText;
                this.classList.remove('disabled');
            }, 1000);
        });
    });
}

function trackClick(productId) {
    // Send click data to analytics endpoint
    fetch('/api/track-click', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            productId: productId,
            timestamp: new Date().toISOString(),
            userAgent: navigator.userAgent,
            referrer: document.referrer
        })
    }).catch(error => {
        console.log('Analytics tracking failed:', error);
    });
}

// Auto-refresh dashboard data
function initializeAutoRefresh() {
    if (window.location.pathname === '/dashboard' || window.location.pathname === '/analytics') {
        // Refresh every 5 minutes
        setInterval(() => {
            refreshDashboardData();
        }, 300000);
    }
}

function refreshDashboardData() {
    // Show loading indicator
    showLoadingIndicator();
    
    // Refresh page data
    setTimeout(() => {
        location.reload();
    }, 1000);
}

function showLoadingIndicator() {
    const indicator = document.createElement('div');
    indicator.id = 'loading-indicator';
    indicator.className = 'position-fixed top-0 end-0 m-3 p-3 bg-primary text-white rounded shadow';
    indicator.innerHTML = `
        <div class="d-flex align-items-center">
            <div class="spinner-border spinner-border-sm me-2" role="status"></div>
            <span>Memperbarui data...</span>
        </div>
    `;
    
    document.body.appendChild(indicator);
    
    // Remove after 3 seconds
    setTimeout(() => {
        const existingIndicator = document.getElementById('loading-indicator');
        if (existingIndicator) {
            existingIndicator.remove();
        }
    }, 3000);
}

// Utility functions
function formatCurrency(amount) {
    return new Intl.NumberFormat('id-ID', {
        style: 'currency',
        currency: 'IDR',
        minimumFractionDigits: 0
    }).format(amount);
}

function formatNumber(number) {
    return new Intl.NumberFormat('id-ID').format(number);
}

function formatDate(date) {
    return new Intl.DateTimeFormat('id-ID', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    }).format(new Date(date));
}

// Copy to clipboard function
function copyToClipboard(text) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(text).then(() => {
            showNotification('Link berhasil disalin!', 'success');
        }).catch(() => {
            fallbackCopyToClipboard(text);
        });
    } else {
        fallbackCopyToClipboard(text);
    }
}

function fallbackCopyToClipboard(text) {
    const textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    
    try {
        document.execCommand('copy');
        showNotification('Link berhasil disalin!', 'success');
    } catch (err) {
        showNotification('Gagal menyalin link', 'error');
    }
    
    document.body.removeChild(textArea);
}

// Show notification
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

// Export functions for global use
window.copyToClipboard = copyToClipboard;
window.showNotification = showNotification;
window.formatCurrency = formatCurrency;
window.formatNumber = formatNumber;
window.formatDate = formatDate;

// Handle form submissions with loading states
document.addEventListener('submit', function(event) {
    const form = event.target;
    const submitButton = form.querySelector('button[type="submit"]');
    
    if (submitButton) {
        const originalText = submitButton.innerHTML;
        submitButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
        submitButton.disabled = true;
        
        // Re-enable button after 5 seconds (fallback)
        setTimeout(() => {
            submitButton.innerHTML = originalText;
            submitButton.disabled = false;
        }, 5000);
    }
});

// Handle AJAX errors globally
window.addEventListener('unhandledrejection', function(event) {
    console.error('Unhandled promise rejection:', event.reason);
    showNotification('Terjadi kesalahan. Silakan coba lagi.', 'error');
});

// Performance monitoring
if ('performance' in window) {
    window.addEventListener('load', function() {
        setTimeout(() => {
            const perfData = performance.getEntriesByType('navigation')[0];
            if (perfData) {
                console.log('Page load time:', perfData.loadEventEnd - perfData.loadEventStart, 'ms');
            }
        }, 0);
    });
}